// soft leon pour ... bouger des voitures dans des casiers
// remarks : all libs to add to ardino IDE in  in crab's drive/Leon/Libs

// a kind of release tag
#define HARDWARE_NAME "20150404_LEON"

// matos :
//  6 MD10C
//  3 arduino uno
//  3 shield proto DFRobot
//  5 strip 30 leds ws2812
//  3 cartes rs485

#ifdef ARDUINO_AVR_LEONARDO
// romeo/leonardo specific
// http://www.dfrobot.com/wiki/index.php/Romeo_V2-All_in_one_Controller_%28R3%29_%28SKU:DFR0225%29
// requires rs-485 modifications (RS485-Arduino-Library-Leon/RS485_Leon.h) or pin 4 is shared between motor1 and rs
// Digital 4	Motor 1 Direction control
// Digital 5	Motor 1 PWM control
// Digital 6	Motor 2 PWM control
// Digital 7	Motor 2 Direction control
// drived the same way than MD10C, use MotorMD10CRun( 6 /* pinPwm */ ,  7 /* pinRotation */ , xx /*Dir */,  yy /* SpeedValue*/ ) ;
#endif

#if defined(__AVR_ATmega1280__) || defined(__AVR_ATmega2560__)
  #define ARDU_MEGA
#endif

#include <SoftwareSerial.h>
#include <EEPROM.h>

// TODO_LATER :  cablage  http://fritzing.org/home/

// rs485 http://www.lctech-inc.com pins RX_PIN 2->DI, TX_PIN 3->RO, ENABLE_PIN 10->DE+RE, +5V->VCC GND->GND
// maxMsgLen 20 - http://wiki.3615senor.org/doku.php?id=rs485
// https://github.com/Protoneer/RS485-Arduino-Library
// remarks : works on direct link with two arduinos too - wire 2<->3, 3<->2, GND<->GND
//#include <RS485_Leon.h>
#define STX 2
#define ETX 3
#define RX_PIN 2
#define TX_PIN 3
#define ENABLE_PIN 10
#define maxMsgLen 20
SoftwareSerial RS485_Line( RX_PIN, TX_PIN);

// NeoPixel strip led ws2812 https://github.com/adafruit/Adafruit_NeoPixel
#include <Adafruit_NeoPixel.h>
#include <avr/power.h>

// light ... max 80 leds in ARMOIRE2 (50 light+30 Feu+securitee anti bugs)
// ACT_CAS5_LEDFEU
#define NEOPIXEL_NUMPIXELS      (80+2)
#define LED_FEU_START 50
#define LED_CAS1_LEDCOLOR_START 10

// change the two above to tune global led brightness
#define INTENSITE_CASIER 1.0
#define INTENSITE_FEU 1.0

float intensiteLumiereCasier=INTENSITE_CASIER;
float intensiteLumiereFeu=INTENSITE_FEU;

// defaut role for arduino is "UNSET"(a bad configured arduino won't sent unexpected on RS485)
// change this by soft, type "2R" in serial monitor and see the module tells you it changes
#define ROLE_DEFAULT 0
// level of sent to the debug line, 0 - none, 2 all
#define DBG_VERBOSE_DEFAULT 0

// --- role section, i.e. the place the arduino is in the global scheme
typedef enum {
   ROLE_UNSET    //0
 , ROLE_ARMOIRE1 //1
 , ROLE_REGIE    //2
 , ROLE_ARMOIRE2 //3
 , ROLE_MAX // max supported by this revision
} RoleList_te;

#define RoleGet() EEPROM.read(2)
#define RoleSet( R) EEPROM.write(2, R)
int Role; // readed a lot... store here to read less from eeprom

// all the function (actions) we want to perform, switch, light, motor, & so on
typedef enum {
   ACT_UNKNOWN        //  0
 , ACT_TESTALIVE      //  1
  // in ROLE_ARMOIRE1
 , ACT_CAS2_MOTOR
 , ACT_CAS1_MOTOR
 , ACT_CAS1_LEDCOLOR
 , ACT_CAS3_MOTORCAR 
 , ACT_CAS2_FADELED
 , ACT_RFU2
 , ND_ARMOIRE1
  // in ROLE_ARMOIRE2
 , ACT_CAS4_RELAIS 
 , ACT_CAS0_LEDONOFF
 , ACT_CAS5_LEDFEU
 , ACT_CAS6_MOTOR1
 , ACT_CAS6_MOTOR2 
 , ACT_PM_MOTOR
 , ND_ARMOIRE2
} ActList_te;

int TESTALIVE_Count = 0;
unsigned long TESTALIVE_Millis = 0;
// send only if changed
int CAS2_MOTORG_B1state = HIGH; 
int CAS2_MOTORD_B2state = HIGH;
int CAS1_MOTOR_SWstate = HIGH;
int CAS3_MOTORCAR_Pstate = 0;
int CAS2_FADELED_Pstate = 0; // ACT_CAS2_FADELED

int CAS4_RELAIS_BState = HIGH;
int CAS5_LEDFEU_BState = HIGH; // not pressed
int CAS0_LEDONOFF_BState = HIGH;
int CAS6_MOTOR1G_BState = HIGH;
int CAS6_MOTOR1D_BState = HIGH;
int CAS6_MOTOR2_BState = HIGH;
int PM_BState = HIGH;


// message exchanged thru RS485
char Message[maxMsgLen+1+3] ;

uint32_t DbgNum = 0; // 0..999999999

// dev 

// level of sent to the debug line
int DbgVerbose = DBG_VERBOSE_DEFAULT;
// blink on activity
int DbgBlink = 2;
// inject some false commands from command line :)
int DbgDeviceId = 0;
int DbgSensorValue = 0;


// simple mecanism to re-send and gain fiability
#define RS_RETRY 5
unsigned int RsFrameCount = 0;
unsigned long FrameCountAll = 0;
unsigned long FrameCountErr = 0;

void dbgprintf(char *fmt, ... ){
  char buf[128]; // resulting string limited to 128 chars
  va_list args;
  va_start (args, fmt );
  vsnprintf(buf, 128, fmt, args);
  va_end (args);
  Serial.print(buf);
}

#define dbgReadCh Serial.read

// check values in eprom are not all madness... first boot they are supposed to be 255
void EpromSanity() {
  if( RoleGet() >= ROLE_MAX) {
    RoleSet( ROLE_UNSET);
  }
  Role = RoleGet();
}

// show eprom content on debug line (a kind of help too
void EpromDump(){
  Role = RoleGet();
  dbgprintf("LeonSoft, Ben Farey, Crab rev %s, 2015\r\n", HARDWARE_NAME);
  #ifdef ARDU_MEGA
    dbgprintf( "for arduino mega \r\n", Role);
  #endif
  dbgprintf( "9R  :Role %i, ", Role);
  switch( Role) {
    case ROLE_UNSET :
      dbgprintf( " not set... try 1R 2R ... %iR", ROLE_MAX-1);
      break;
    case ROLE_REGIE :
      dbgprintf( " REGIE");
      break;
    case ROLE_ARMOIRE1 :
      dbgprintf( " ARMOIRE1");
      break;
    case ROLE_ARMOIRE2 :
      dbgprintf( " ARMOIRE2");
      break;
  }
  dbgprintf("\r\n");
  dbgprintf( "9v  :verbose %i \r\n", DbgVerbose);
  dbgprintf( "Rs385 Errs %lu", FrameCountErr);
  dbgprintf( "/%lu\r\n", FrameCountAll);
}

// motors, lots all the same scheme
int MotorMD10CRun( int pinPwm, int pinRotation, int Dir, int SpeedValue) { 
  if(Dir > 0) {
    digitalWrite( pinRotation, HIGH);             // turn in one direction
  } else {
    digitalWrite( pinRotation, LOW);             // turn in ... another direction
  }
  analogWrite( pinPwm , SpeedValue);
  if (DbgVerbose >= 1) dbgprintf( "MD10 %i %i\r\n", Dir, SpeedValue);
  return(0);
}

int MF_Act = -1;
unsigned long MF_Millis=0; // a kind of timer
int MF_pinPwm = 0;
int MF_pinRotation = 0;
int MF_Times[3];
int MF_Pwm[3];

// follow instructions from MotorFollow without blocking the the software execution
int MotorFollowRun( ){
  if (MF_Act >= 0) {
    if ( MF_Millis < millis()) { // Act
      MF_Act--;
      if (MF_Act < 0) { //  this is the end
        MotorMD10CRun( MF_pinPwm, MF_pinRotation, 0 /* Dir */, 0 /* Speed */);
        MF_pinPwm = 0;
      } else {
        if (MF_Pwm[MF_Act] > 0) {
          MotorMD10CRun( MF_pinPwm, MF_pinRotation, 0 /* Dir */, MF_Pwm[MF_Act] /* Speed */);
        } else {
          MotorMD10CRun( MF_pinPwm, MF_pinRotation, 1 /* Dir */, -MF_Pwm[MF_Act] /* Speed */);
        }
      }
      MF_Millis = millis() + MF_Times[MF_Act];
    }
  }
}

// give power/time, power/time , power/time to follow for a MD10 motor. ends with motor stop
// MotorMD10CRun will do the real job
// drive only one motor at a time, new call to MotorFollow stop the crrent move in the middle of it and start with P1/T1
// negative powers to drive PinRotation and rotate in reverse direction
int MotorFollow( int pinPwm, int pinRotation, int P1, int T1, int P2, int T2, int P3, int T3){
  
  if (0 != MF_pinPwm) { // first, stop a previous command
    MotorMD10CRun( MF_pinPwm, MF_pinRotation, 0 /* Dir */, 0 /* Speed */);
  }

  if (0 == pinPwm) { // asked to stop the current run
    MF_Act = -1;
    return(0);
  }
  
  MF_pinPwm = pinPwm;
  MF_pinRotation = pinRotation;
  MF_Times[2] = T1;
  MF_Times[1] = T2;
  MF_Times[0] = T3;
  MF_Pwm[2] = P1;
  MF_Pwm[1] = P2;
  MF_Pwm[0] = P3;
  MF_Act = 3;
  MF_Millis = millis(); // MotorFollowRun will run
  return(0);
}


// Pixelm section
Adafruit_NeoPixel pixels = Adafruit_NeoPixel( NEOPIXEL_NUMPIXELS, 5/*NEOPIXEL_PIN (default)*/, NEO_GRB + NEO_KHZ800);

// change pixels [PixSt..PixNd[ Color to RGB values, from 0,0,0 up to 255,255,255
int LightNeopixel( uint16_t PixSt, uint16_t PixNd, uint8_t r, uint8_t g, uint8_t b) {
  for(int i=PixSt;i<=PixNd;i++){
    // pixels.Color takes RGB values, from 0,0,0 up to 255,255,255

//modif OH
//    pixels.setPixelColor(i, pixels.Color( (byte) intensiteLumiereCasier*r, (byte) intensiteLumiereCasier*g, (byte) intensiteLumiereCasier*b)); // Moderately bright green color.
    pixels.setPixelColor(i, pixels.Color( (byte) intensiteLumiereCasier*r, (byte) intensiteLumiereCasier*g, (byte) intensiteLumiereCasier*b)); // Moderately bright green color.
//fin modif OH
  }

  pixels.show(); // This sends the updated pixel color to the hardware.
  return(0);
}


// Fiery demon horns (rawr!) for Adafruit Trinket/Gemma.
// Adafruit invests time and resources providing this open source code,
// please support Adafruit and open-source hardware by purchasing
// products from Adafruit!

// TODO_LATER : N_HORNS*N_LEDS must be 30
#define N_HORNS 6
#define N_LEDS 5 // Per horn

// /\ -> Fire-like effect is the sum of multiple triangle
// ____/ \____ waves in motion, with a 'warm' color map applied.
#define N_WAVES 6 // Number of simultaneous waves (per horn)
// Coordinate space for waves is 16x the pixel spacing,
// allowing fixed-point math to be used instead of floats.
struct {
  int16_t lower; // Lower bound of wave
  int16_t upper; // Upper bound of wave
  int16_t mid; // Midpoint (peak) ((lower+upper)/2)
  uint8_t vlower; // Velocity of lower bound
  uint8_t vupper; // Velocity of upper bound
  uint16_t intensity; // Brightness at peak
} wave[N_HORNS][N_WAVES];
long fade; // Decreases brightness as wave moves

// Gamma correction improves appearance of midrange colors
const uint8_t gamma[] PROGMEM = {
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2,
  2, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 5, 5, 5,
  5, 6, 6, 6, 6, 7, 7, 7, 7, 8, 8, 8, 9, 9, 9, 10,
  10, 10, 11, 11, 11, 12, 12, 13, 13, 13, 14, 14, 15, 15, 16, 16,
  17, 17, 18, 18, 19, 19, 20, 20, 21, 21, 22, 22, 23, 24, 24, 25,
  25, 26, 27, 27, 28, 29, 29, 30, 31, 32, 32, 33, 34, 35, 35, 36,
  37, 38, 39, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 50,
  51, 52, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 66, 67, 68,
  69, 70, 72, 73, 74, 75, 77, 78, 79, 81, 82, 83, 85, 86, 87, 89,
  90, 92, 93, 95, 96, 98, 99, 101, 102, 104, 105, 107, 109, 110, 112, 114,
  115, 117, 119, 120, 122, 124, 126, 127, 129, 131, 133, 135, 137, 138, 140, 142,
  144, 146, 148, 150, 152, 154, 156, 158, 160, 162, 164, 167, 169, 171, 173, 175,
  177, 180, 182, 184, 186, 189, 191, 193, 196, 198, 200, 203, 205, 208, 210, 213,
  215, 218, 220, 223, 225, 228, 231, 233, 236, 239, 241, 244, 247, 249, 252, 255
};

static void random_wave(uint8_t h, uint8_t w) { // Randomize one wave struct
  wave[h][w].upper = -1; // Always start just below head of strip
  wave[h][w].lower = -16 * (3 + random(4)); // Lower end starts ~3-7 pixels back
  wave[h][w].mid = (wave[h][w].lower + wave[h][w].upper) / 2;
  wave[h][w].vlower = 3 + random(4); // Lower end moves at ~1/8 to 1/4 pixel/frame
  wave[h][w].vupper = wave[h][w].vlower + random(4); // Upper end moves a bit faster, spreading wave
  wave[h][w].intensity = 300 + random(600);
}

void FireSetup() {
  uint8_t h, w;

  randomSeed(analogRead(1));
  pixels.begin();
  for (h = 0; h < N_HORNS; h++) {
    for (w = 0; w < N_WAVES; w++) random_wave(h, w);
  }
  fade = 234 + N_LEDS / 2;
  if (fade > 255) fade = 255;
}

void fire() {
  uint8_t h, w, i, r, g, b;
  int16_t x;
  uint16_t sum;

  for (h = 0; h < N_HORNS; h++) { // For each horn...
    for (x = 7, i = 0; i < N_LEDS; i++, x += 16) { // For each LED along horn...
      for (sum = w = 0; w < N_WAVES; w++) { // For each wave of horn...
        if ((x < wave[h][w].lower) || (x > wave[h][w].upper)) continue; // Out of range
        if (x <= wave[h][w].mid) { // Lower half of wave (ramping up to peak brightness)
          sum += wave[h][w].intensity * (x - wave[h][w].lower) / (wave[h][w].mid - wave[h][w].lower);
        } else { // Upper half of wave (ramping down from peak)
          sum += wave[h][w].intensity * (wave[h][w].upper - x) / (wave[h][w].upper - wave[h][w].mid);
        }
      }
      // Now the magnitude (sum) is remapped to color for the LEDs.
      // A blackbody palette is used - fades white-yellow-red-black.
      if (sum < 255) { // 0-254 = black to red-1
        r = pgm_read_byte(&gamma[sum]);
        g = b = 0;
      } else if (sum < 510) { // 255-509 = red to yellow-1
        r = 255;
        g = pgm_read_byte(&gamma[sum - 255]);
        b = 0;
      } else if (sum < 765) { // 510-764 = yellow to white-1
        r = g = 255;
        b = pgm_read_byte(&gamma[sum - 510]);
      } else { // 765+ = white
        r = g = b = 255;
      }

//modif OH
      //pixels.setPixelColor(LED_FEU_START+ (h * N_LEDS) + i, r,g,b);
      pixels.setPixelColor(LED_FEU_START+ (h * N_LEDS) + i, (byte) intensiteLumiereFeu*r, (byte) intensiteLumiereFeu*g, (byte) intensiteLumiereFeu*b);
//fin modif OH
    }

    for (w = 0; w < N_WAVES; w++) { // Update wave positions for each horn
      wave[h][w].lower += wave[h][w].vlower; // Advance lower position
      if (wave[h][w].lower >= (N_LEDS * 16)) { // Off end of strip?
        random_wave(h, w); // Yes, 'reboot' wave
      } else { // No, adjust other values...
        wave[h][w].upper += wave[h][w].vupper;
        wave[h][w].mid = (wave[h][w].lower + wave[h][w].upper) / 2;
        wave[h][w].intensity = (wave[h][w].intensity * fade) / 256; // Dimmer
      }
    }
  }
  pixels.show();
}


int RS485_Send( char* Msg, int Len) {
  byte Ch;
  int Count;
  byte Crc = 0;
  
  digitalWrite ( ENABLE_PIN, HIGH);  // enable sending
  delay(1); // wait line establish firmly

  RS485_Line.write( (byte)STX);
  Count = 0;
  Crc = 0; // not a true CRC, just a check
  while(Count < Len) {
    RS485_Line.write( Msg[Count]);
    Crc = Crc + Msg[Count];
    Count++;
  }
  RS485_Line.write( Crc);
  RS485_Line.write( (byte)ETX);
  
  delay(1); // so said to wait for byte to sent... probably not usefull
  digitalWrite ( ENABLE_PIN, LOW);  // disable sending  
}

int RS485_Recv( char* Msg, int MaxMsg) {
  int WaitCount;
  byte Ch;
  byte Ch1;
  int Recv;
  byte Crc;
  
  // wait STX but not too long
  if (RS485_Line.available() <= 0) {
    delay(1);
  }
  if (RS485_Line.available() <= 0) {
    return( -1); // nothing
  }
  Ch = RS485_Line.read();
  if(STX != Ch) {
    return(-1); // not my frame
  }
    
  Recv = 0;
  Crc = 0;
  Ch = 0; Ch1 = 0;
  while( Recv < MaxMsg) {
    if (RS485_Line.available() <= 0) {
      delay(1);
    }
    if (RS485_Line.available() <= 0) {
      return( -3); // timeout
    }
    Ch = RS485_Line.read();
    if (ETX == Ch) { // received
       Ch = Msg[Recv-1];
       if ((Recv > 0) && (Ch == Crc)) {
          Msg[Recv] = 0; // end string for debug purpose, if it's text we can print it as-is
          return( Recv-1); // Good
       } else {
         dbgprintf( "warn Crc %i %i\r\n", Ch, Crc);
       }
    }
    Msg[Recv++] = Ch;
    Crc = Crc+Ch1; // do not add crc to crc
    Ch1 = Ch;
  }
  return( -2);
}
  
// send message to RS485 - call this from ActByRole/regie
#define SEND_RTRY 10
#define SEND_ACK 5
// 10 ms by ACK wait means 500ms blocking at worst
int SendMessage( int DeviceId, int sensorValue) {

  int Resend = SEND_RTRY;
  int AckWait;
  int FrameSeq;

  //RsFrameCount = (RsFrameCount+1)%10000;
  RsFrameCount ++;
  if (RsFrameCount > 99) {
    RsFrameCount = 0;
  }
  FrameCountAll++;

  while (Resend > 0) {
    Resend--;

    //sprintf(Message, "Msg%3.3i%3.3i%2.2i", DeviceId, sensorValue, RsFrameCount);
    strcpy(Message,"Msg11122233X");
    Message[3] = '0'+(DeviceId/100)%10;
    Message[4] = '0'+(DeviceId/ 10)%10;
    Message[5] = '0'+(DeviceId    )%10;
    Message[6] = '0'+(sensorValue/100)%10;
    Message[7] = '0'+(sensorValue/ 10)%10;
    Message[8] = '0'+(sensorValue    )%10;
    Message[ 9] = '0'+(RsFrameCount/ 10)%10;
    Message[10] = '0'+(RsFrameCount    )%10;

    if (RS485_Send( Message, 11))
    {
      if (DbgVerbose >= 3) dbgprintf("Sending1:%s\r\n", Message);
      
      AckWait = SEND_ACK;
      while(AckWait > 0) {
        if( RS485_Recv( Message, maxMsgLen) > 0)
        {
          if (DbgVerbose >= 3) dbgprintf( "Received1:%i %s\r\n", AckWait, Message);
          if('A' == Message[0]) {
            FrameSeq = 0;
            FrameSeq += (Message[ 3]-'0')* 10; 
            FrameSeq += (Message[ 4]-'0');
            //dbgprintf( "Ack 2:%i %i\r\n", FrameSeq, RsFrameCount);
            if (FrameSeq == RsFrameCount%100) { // we get the expected Ack
              if (DbgVerbose >= 2) dbgprintf( "Send Try %i Ack %i \r\n", SEND_RTRY-Resend, SEND_ACK-AckWait);
              TESTALIVE_Millis = millis(); // delay next live check
              return( 0);
            }
          }
        } else {
          if (DbgVerbose >= 2) {
            dbgprintf( "Warn NoAck %i ", RsFrameCount);
            dbgprintf( "%lu ms\r\n", millis());
          }
        }
        AckWait--;
        if (0 != AckWait) {
          delay(10);
        }
      }
    } else {
      dbgprintf("Err Send %i\r\n", RsFrameCount);
    } 
  }
  FrameCountErr++;
  dbgprintf( "Err NoAck %i \r\n", RsFrameCount);
  dbgprintf( "at %lu ms\r\n", millis());
  return(-1);
}

// send message to RS485
int SendAck( int FrameSeq) {

  int Resend = RS_RETRY;

  //sprintf(Message, "Ack%2.2i", RsFrameCount);
  strcpy(Message,"Ack33");
  Message[ 3] = '0'+(FrameSeq/ 10)%10;
  Message[ 4] = '0'+(FrameSeq    )%10;

  if (RS485_Send( Message, 6))
  {
    if (DbgVerbose >= 3) dbgprintf("Sending2:%s %i\r\n", Message, ENABLE_PIN);
  } else {
    dbgprintf("ErrSendAck\r\n");
  } 
  return(0);
}

// extract info from a received message
int MessageDecode( char* Message, int* pDeviceId, int* psensorValue, int* pFrameSeq) {
  
  *pDeviceId = 0;
  *pDeviceId += (Message[3]-'0')*100; 
  *pDeviceId += (Message[4]-'0')* 10; 
  *pDeviceId += (Message[5]-'0'); 
  *psensorValue = 0;
  *psensorValue += (Message[6]-'0')*100; 
  *psensorValue += (Message[7]-'0')* 10; 
  *psensorValue += (Message[8]-'0');
  *pFrameSeq = 0;
  *pFrameSeq += (Message[ 9]-'0')* 10; 
  *pFrameSeq += (Message[10]-'0');
      
  return( true);
}

int MessageIsReady( int* pDeviceId, int* psensorValue) {
  
  int FrameSeq = 0;

  // debug command
  if (0 != DbgDeviceId) {
    *pDeviceId = DbgDeviceId;
    *psensorValue = DbgSensorValue;
    DbgDeviceId = 0;
    DbgSensorValue = 0;
    return( true);
  }

  if (RS485_Recv( Message, maxMsgLen) > 0)
  //if (RS485_ReadPlainMessage( fAvailable, fRead, Message))
  {
    if (DbgVerbose >= 3) dbgprintf( "Received2:%s\r\n", Message);
    if ('A' == Message[0]) { // it's an ack, probab from someone else
      return( false);
    }
    MessageDecode( Message, pDeviceId, psensorValue, &FrameSeq);
    switch(Role) {
      case ROLE_ARMOIRE1:
        if ((*pDeviceId > 0) && (*pDeviceId <= ND_ARMOIRE1)) {
          SendAck( FrameSeq);
        }
        break;
      case ROLE_ARMOIRE2:
        if ((*pDeviceId > ND_ARMOIRE1) && (*pDeviceId <= ND_ARMOIRE2)) {
          SendAck( FrameSeq);
        } else if (*pDeviceId == ACT_TESTALIVE) {
          SendAck( FrameSeq);
        }
        break;
    }

    if (RsFrameCount != FrameSeq) { // it's a new frame, not a retry
      RsFrameCount = FrameSeq;
      return( true);
    } // else it's a retry of an old frame, ignore it
  }

  return( false);
}

// command (debug) enter here
void CmdLineParse( unsigned char Ch) {
  char EprSet = 0;

  switch(Ch) {
      case 'R' : // ROLE_ .. regie, casierX ....
        RoleSet(DbgNum%10); 
        EprSet=1; 
        break;
      case 'h' :
      case 'H' :
        EprSet=1; // will provoque EpromDump ... a kind of help
        break;
      case 'v' : // verbose level
        DbgVerbose = DbgNum%10;
        EprSet=1; // just to show something is done on serial line
        break;
     case 'T': // test function, send to friends from command line xxxyyyT
        DbgDeviceId = (DbgNum/1000)%1000;
        DbgSensorValue = (DbgNum)%1000;
        dbgprintf("debug Device:%i Value:%i\r\n", DbgDeviceId, DbgSensorValue);
        if (ROLE_REGIE == Role) {
          SendMessage( DbgDeviceId, DbgSensorValue);
        }
        break;
     default :
        if((Ch>='0')&&(Ch<='9')) {
          DbgNum = DbgNum%100000000; // 400000000 --> 8 digits 0..9
          DbgNum = DbgNum*10+Ch-'0';
        }
        break;
  }
  if (1 == EprSet){
      EpromSanity();
      EpromDump();
      EprSet = 0;
  }
}

// get a 0 zone and a full zone and map 1024 readings to 255 actions
int PotardMap (int sensorValue) {
  int ToSend;
  
  // A0 0..1024
  if (sensorValue < 100) {
    ToSend = 0;
  } else if (sensorValue > 900) {
    ToSend = 255;
  } else {
    ToSend = map( sensorValue, 100, 900, 0, 255);
  }
  return( ToSend);
}

// don't send every moves or that might send on other action on potard vibration 
bool PotardToSend( int sensorValue, int PreviousVal) {
  if ( abs(sensorValue-PreviousVal)> 2)
    return( true);

  if ((sensorValue< 10) && (sensorValue != PreviousVal))
    return( true);

  return( false);
}

// blink led to tell it's alive!
void Activity( void) {
  digitalWrite(13, HIGH);   // turn the LED on (HIGH is the voltage level)
  delay(400);
  digitalWrite(13, LOW);
  delay(100);
}

// ACT_TESTALIVE
void ActTestAlive( int sensorValue) {
  TESTALIVE_Count = sensorValue;
  switch(TESTALIVE_Count %5) {
    case 0 :
      if (DbgVerbose >= 1) dbgprintf( "TESTALIVE %3.3i %3.3i \r\n", ACT_TESTALIVE, sensorValue);
      break;
    case 1 :
      digitalWrite(13, HIGH);   // turn the LED on (HIGH is the voltage level)
      break;
    case 2 :
      digitalWrite(13, LOW);
      break;
    }
}

void InitByRole (){
  // here init for each Role
  switch(Role) {
    case ROLE_REGIE :
  // REGIE in ARMOIRE1
      // all switch link to ground if pressed
      // , ACT_CAS2_MOTORG
      pinMode(  9, INPUT_PULLUP); 
      CAS2_MOTORG_B1state = digitalRead(9); 
      // , ACT_CAS2_MOTORD
      pinMode( 11, INPUT_PULLUP); 
      CAS2_MOTORD_B2state = digitalRead(11);
      // , ACT_CAS1_MOTORG
      // , ACT_CAS1_MOTORD
      // , ACT_CAS1_LEDCOLOR
      pinMode( 12, INPUT_PULLUP); 
      CAS1_MOTOR_SWstate = digitalRead(12);
      // , ACT_CAS3_MOTORCAR 
      pinMode( A0, INPUT_PULLUP); 
      CAS3_MOTORCAR_Pstate = PotardMap(analogRead(A0));
      // ACT_CAS2_FADELED
      pinMode( A3, INPUT_PULLUP); 
      CAS2_FADELED_Pstate = PotardMap(analogRead(A3));
  // REGIE in ARMOIRE2
      // , ACT_CAS4_RELAIS
      pinMode( 4, INPUT_PULLUP);
      CAS4_RELAIS_BState = digitalRead( 4);
      //  a Button2position  to drive ACT_CAS5_LEDFEU is wired on pin 5 of arduino REGIE
      pinMode( 5, INPUT_PULLUP); 
      CAS5_LEDFEU_BState = digitalRead(5);
      // , ACT_CAS0_LEDONOFF
      pinMode( A2, INPUT_PULLUP); 
      CAS0_LEDONOFF_BState = digitalRead( A2);
      // , ACT_CAS6_MOTOR1
      pinMode( 6, INPUT_PULLUP); 
      CAS6_MOTOR1G_BState = digitalRead( 6);
      pinMode( 7, INPUT_PULLUP); 
      CAS6_MOTOR1D_BState = digitalRead( 7);
      // , ACT_CAS6_MOTOR2
      pinMode( 8, INPUT_PULLUP); 
      CAS6_MOTOR2_BState = digitalRead( 8);
      // , ACT_PM_MOTOR
      pinMode( A1, INPUT_PULLUP); 
      PM_BState = digitalRead( A1);
      break;

    case ROLE_ARMOIRE1 : // in ARMOIRE1
// , ACT_CAS2_MOTORG, ACT_CAS2_MOTORD motor wired to pins 5(pwm) and 4(Rotation) of arduino ARMOIRE1
      pinMode(5,OUTPUT);
      pinMode(4,OUTPUT);
// , ACT_CAS1_MOTORG, ACT_CAS1_MOTORD motor wired to pins 7(pwm) and 6(Rotation) of arduino ARMOIRE1
      pinMode(7,OUTPUT);
      pinMode(6,OUTPUT);
// , ACT_CAS3_MOTORCAR motor wired to pins 9(pwm) and 8(Rotation) of arduino ARMOIRE1
      pinMode(9,OUTPUT);
      pinMode(8,OUTPUT);
// , ACT_CAS1_LEDCOLOR, ACT_CAS2_FADELED
      // example a strip led is wired on arduino "CASIER1" pin 12
      pixels.setPin(12);
      pixels.begin(); 
      LightNeopixel( 0, 19, 255, 255, 224); // ACT_CAS2_FADELED Light Yellow
      LightNeopixel( 20, 28, 124, 252, 0); // Lame Yellow
      LightNeopixel( 29, 38, 255, 255, 224); // Light Yellow
       
      break;

    case ROLE_ARMOIRE2 : // in ARMOIRE2
      // , ACT_CAS4_RELAIS
      pinMode( 11,OUTPUT);
      // , ACT_CAS6_MOTOR1 motor wired to pins 5(pwm) and 4(Rotation) of arduino ARMOIRE2
      pinMode(  4,OUTPUT); // Dir
      pinMode(  5,OUTPUT); // PWM
      // , ACT_CAS6_MOTOR2    // 15
      pinMode(  6, OUTPUT); // PWM
      pinMode(  7, OUTPUT); // Dir
      pinMode( A0, INPUT_PULLUP); // switch to GND to stop rotation
      // , ACT_PM_MOTOR a "Portee Mannteauu" thru MD10 is wired to pins 8(pwm) and 9(Rotation) of arduino ARMOIRE2
      pinMode(  8,OUTPUT); // Dir
      pinMode(  9,OUTPUT); // PWM
      // , ACT_CAS0_LEDONOFF, ACT_CAS5_LEDFEU a strip led is wired on arduino ARMOIRE2 pin 12
      CAS5_LEDFEU_BState=HIGH; // not pressed, use this vriable in regie AND in ARMOIRE2
      pixels.setPin(12);
      pixels.begin();
      FireSetup();
      LightNeopixel( 10, 49, 255, 255, 224); //Light Yellow
      LightNeopixel( 0, 9, 255, 182, 193); //Light Pink
      LightNeopixel( 20, 21, 0, 0, 0);
      LightNeopixel( 28, 29, 0, 0, 0);
      LightNeopixel( 50, 79,   0,   0,   0);
      break;
  }
}

unsigned long LoopMillisSt=0;
unsigned long LoopMillisNd=0;

void ActByRole() {
  int DeviceId = 0;
  int sensorValue = 0;
  int ToSend;
  
  //if (DbgVerbose >= 1) dbgprintf( "TimeLoop= %ims\r\n", LoopMillisNd-LoopMillisSt);
  LoopMillisSt = millis();

  // here actions for each Role
  switch( Role) {
    case ROLE_REGIE : // here the functions to send with the arduino "REGIE"
      // the function we want to perform distantly
      
      // switches to send to ARMOIRE1

      // , ACT_CAS2_MOTOR
      sensorValue = digitalRead(9);
      if (CAS2_MOTORG_B1state != sensorValue) {
          CAS2_MOTORG_B1state = sensorValue;
          if (LOW == sensorValue) { // pressed
            //dbgprintf("ACT_CAS2_MOTOR B1\r\n");
            SendMessage( ACT_CAS2_MOTOR, 2);
          }
      }
      sensorValue = digitalRead(11);
      if (CAS2_MOTORD_B2state != sensorValue) {
          CAS2_MOTORD_B2state = sensorValue;
          if (LOW == sensorValue) { // pressed
            //dbgprintf("ACT_CAS2_MOTOR B2\r\n");
            SendMessage( ACT_CAS2_MOTOR, 1);
          } else {
            SendMessage( ACT_CAS2_MOTOR, 0);
          }
      }
      // , ACT_CAS1_MOTOR
      // , ACT_CAS1_LEDCOLOR
      sensorValue = digitalRead(12);
      if ( CAS1_MOTOR_SWstate != sensorValue) {
          CAS1_MOTOR_SWstate = sensorValue;
          // dbgprintf("ACT_CAS1_MOTOR %i", sensorValue);
          SendMessage( ACT_CAS1_MOTOR, sensorValue);
          // dbgprintf("ACT_CAS1_LEDCOLOR %i", sensorValue);
          SendMessage( ACT_CAS1_LEDCOLOR, sensorValue);
      }

      // , ACT_CAS3_MOTORCAR 
      sensorValue = PotardMap(analogRead(A0));
      if ( PotardToSend( sensorValue, CAS3_MOTORCAR_Pstate)) {
        CAS3_MOTORCAR_Pstate = sensorValue;
        if (DbgVerbose >= 1) dbgprintf("CAS3_MOTORCAR %i\r\n", sensorValue);
        SendMessage( ACT_CAS3_MOTORCAR, sensorValue);
      } 
      
      // ACT_CAS2_FADELED
      sensorValue = PotardMap( analogRead(A3));
      if ( PotardToSend( sensorValue, CAS2_FADELED_Pstate)) {
        CAS2_FADELED_Pstate = sensorValue;
        if (DbgVerbose >= 1) dbgprintf("CAS2_FADELED %i\r\n", sensorValue);
        SendMessage( ACT_CAS2_FADELED, sensorValue);
      } 
      
      // end switches for ARMOIRE1
      
      // switches to send to ARMOIRE2

      // , ACT_CAS4_RELAIS
      sensorValue = digitalRead(4);
      if ( CAS4_RELAIS_BState != sensorValue) {
          CAS4_RELAIS_BState = sensorValue;
          //dbgprintf("CAS4_RELAIS %i", sensorValue);
          SendMessage( ACT_CAS4_RELAIS, sensorValue);
      }
      // , ACT_CAS0_LEDONOFF
      sensorValue = digitalRead(A2);
      if ( CAS0_LEDONOFF_BState != sensorValue) {
          CAS0_LEDONOFF_BState = sensorValue;
          //dbgprintf("ACT_CAS0_LEDONOFF %i", sensorValue);
          SendMessage( ACT_CAS0_LEDONOFF, sensorValue);
      }

      // ACT_CAS5_LEDFEU 
      sensorValue = digitalRead(5);
      if (CAS5_LEDFEU_BState != sensorValue) {
          CAS5_LEDFEU_BState = sensorValue;
          //dbgprintf("CAS5_LEDFEU %i\r\n", sensorValue);
          SendMessage( ACT_CAS5_LEDFEU, sensorValue);
      }
      // , ACT_CAS6_MOTOR1
      ToSend = 0;
      sensorValue = digitalRead(6);
      if (CAS6_MOTOR1G_BState != sensorValue) {
        CAS6_MOTOR1G_BState = sensorValue;
        ToSend = 1;
      }
      sensorValue = digitalRead(7);
      if (CAS6_MOTOR1D_BState != sensorValue) {
        CAS6_MOTOR1D_BState = sensorValue;
        ToSend = 1;
      }
      if (ToSend) {
        // dbgprintf("ACT_CAS6_MOTOR1G %i\r\n", sensorValue == LOW);// LOW on pressed
        //  dbgprintf("ACT_CAS6_MOTOR1D %i\r\n", sensorValue == LOW); // LOW pressed
          if (LOW == CAS6_MOTOR1G_BState) { // pressed
            SendMessage( ACT_CAS6_MOTOR1, 1); // G
            //dbgprintf("1\r\n");
          } else if (LOW == CAS6_MOTOR1D_BState) {
            SendMessage( ACT_CAS6_MOTOR1, 2); //D
            //dbgprintf("2\r\n");
          } else {
              SendMessage( ACT_CAS6_MOTOR1, 0);
            //dbgprintf("0\r\n");
          }
      }
      // , ACT_CAS6_MOTOR2
      sensorValue = digitalRead(8);
      if (CAS6_MOTOR2_BState != sensorValue) {
          CAS6_MOTOR2_BState = sensorValue;
          // dbgprintf("ACT_CAS6_MOTOR2 %i\r\n", LOW == sensorValue);
          SendMessage( ACT_CAS6_MOTOR2, sensorValue);
      }
      // , ACT_PM_MOTOR
      sensorValue = digitalRead(A1);
      if (PM_BState != sensorValue) {
          PM_BState = sensorValue;
          // dbgprintf("ACT_PM_MOTOR %i\r\n", LOW == sensorValue);
          SendMessage( ACT_PM_MOTOR, sensorValue);
      }

      // end switches for ARMOIRE2
      

      // blink led for comm tests
      if (abs(millis() - TESTALIVE_Millis) > 200) {
        TESTALIVE_Millis = millis();
        //dbgprintf("Al %lu\n", TESTALIVE_Millis);
        TESTALIVE_Count = ((TESTALIVE_Count+1)%1000);
        SendMessage( ACT_TESTALIVE, TESTALIVE_Count);
        ActTestAlive( TESTALIVE_Count);
      }

      // delay for not overflowing
      //delay( 10);

      // here add nex detect from "pupitre"
      break;
    case ROLE_ARMOIRE1 : // here the functions to treat with arduino role "CASIER1"
      // ACT_CAS2_MOTOR
      // ACT_CAS1_MOTOR
      MotorFollowRun( );

      if( MessageIsReady( &DeviceId, &sensorValue)) {

        if (DbgVerbose >= 2) {
          dbgprintf( "Receiving Casier1: %i %i\r\n", DeviceId, sensorValue);
        }
        
        switch( DeviceId) {
          case ACT_TESTALIVE:
            ActTestAlive( sensorValue);
            break;
          case ACT_CAS2_MOTOR:
            switch( sensorValue) {
               case 0:
                 MotorFollow(0,0,0,0,0,0,0,0);
                 MotorMD10CRun( 5 /* pinPWM */, 4 /* pinRotation */, 0, 0);
                 break;
               case 1:
                 MF_Act = -1; // just stop the current run
                 MotorMD10CRun( 5 /* pinPWM */, 4 /* pinRotation */, 1 /* Dir */, 100 /* Power */);
                 break;
               case 2:
                 MotorFollow(5,4,250,5600,100,2000,10,1000);
                 break;
               }
            break;
          case ACT_CAS1_LEDCOLOR : //Casier 1 changement valeur couleur
            if (LOW == sensorValue) { // pressed
              // dbgprintf("LedOn\r\n");
              LightNeopixel( 10, 19, 176, 226, 255); // Light Sky Blue 1
            } else { 
              // dbgprintf("LedOff\r\n");
              LightNeopixel( 10, 19, 255, 255, 0); // Yellow
            }
            break;
          case ACT_CAS1_MOTOR :
            if (LOW == sensorValue) { // pressed
               MotorFollow(6,7,250,0000,100,13000,10,1000); //reglages vitesse motor en trois temps
            } else {
               MotorFollow(6,7,-250,0000,-100,13000,-10,1000);
            }
//          MotorMD10CRun( 6, 7, 1, sensorValue);
            break;
          case ACT_CAS3_MOTORCAR:
             if (DbgVerbose >= 1) dbgprintf("CAS3_MOTORCAR %i\r\n", sensorValue);
             MotorMD10CRun( 9 /* pinPWM */, 8 /* pinRotation */, 1, sensorValue);
             break;
          case ACT_CAS2_FADELED:
             if (DbgVerbose >= 1) dbgprintf("CAS2_FADELED %i\r\n", sensorValue);
             LightNeopixel( 0, 9
                          , sensorValue
                          , sensorValue
                          , map(sensorValue, 0, 255, 0, 224)); // ACT_CAS2_FADELED Light Yellow
             break;
      }
      //delay(50);
      break; // end role casier1
    case ROLE_ARMOIRE2 :

      // ACT_PM_MOTOR
      MotorFollowRun( );
      // ACT_CAS5_LEDFEU
      if (LOW == CAS5_LEDFEU_BState) {// LOW on pressed
        fire();
        delay(1); // some time for message to arrive
      }
      // ACT_CAS6_MOTOR2
      if (HIGH == CAS6_MOTOR2_BState) {
        if(LOW == digitalRead(A0)) { // switch pressed
          MotorMD10CRun( 6 /* pinPWM */, 7 /* pinRotation */, 0, 0);
          CAS6_MOTOR2_BState = LOW; // 
        }
      }
      
      if( MessageIsReady( &DeviceId, &sensorValue)) {
        switch(DeviceId) {
          case ACT_TESTALIVE:
            ActTestAlive( sensorValue);
            break;
          case ACT_CAS4_RELAIS:
            if (LOW == sensorValue) { // pressed
              digitalWrite( 11, HIGH);
            } else {
              digitalWrite( 11, LOW);
            }
            break;
          case ACT_CAS0_LEDONOFF:
            if (LOW == sensorValue) { // pressed
              LightNeopixel( 20, 29, 0, 0, 0);
            } else {
              LightNeopixel( 22, 27, 255, 255, 224);
            }
            break;
          case ACT_CAS5_LEDFEU:
            if (sensorValue != CAS5_LEDFEU_BState) {
              CAS5_LEDFEU_BState = sensorValue;
              //dbgprintf("CAS5_LEDFEU %i %i\r\n", DeviceId, sensorValue);
              if (LOW == CAS5_LEDFEU_BState) {// LOW on pressed
                dbgprintf("CAS5_LEDFEU fire\r\n");
                //fire();
              } else {
                dbgprintf("CAS5_LEDFEU off\r\n");
                LightNeopixel( LED_FEU_START, LED_FEU_START+30, 255,255,224);
              }
            }
            break;
          case ACT_CAS6_MOTOR1:
            dbgprintf("ACT_CAS6_MOTOR1 %i\r\n", sensorValue);
            switch( sensorValue) {
              case 0:
                MotorMD10CRun( 5 /* pinPWM */, 4 /* pinRotation */, 0, 0);
                break;
              case 1:
                MotorMD10CRun( 5 /* pinPWM */, 4 /* pinRotation */, 1, 255);
                break;
              case 2:
                MotorMD10CRun( 5 /* pinPWM */, 4 /* pinRotation */, 0, 255);
                break;
            }
            break;
          case ACT_CAS6_MOTOR2:
            CAS6_MOTOR2_BState = sensorValue; // LOW if pressed
            if (LOW == CAS6_MOTOR2_BState) {
              MotorMD10CRun( 6 /* pinPWM */, 7 /* pinRotation */, 0, 255);
            }
            // if HIGH, the reader will stop it
            break;
          case ACT_PM_MOTOR:
            if (LOW == sensorValue) { // presssed
              //dbgprintf( "PM 1\r\n");
              MotorFollow(9,8,250,4200,100,1000,10,1000);
            } else {
              //dbgprintf( "PM 0\r\n");
              MotorFollow(9,8,-250,3500,-100,2000,-10,1000);
            }
            break;
          } // end deviceid
        } // end message
      } // end ARMOIRE2
      break;
    // here add new Role (i.e. new arduino receivers support, between case xxx: ... break;
    default :
      dbgprintf( "unsupported role %i, please set <role number>R\r\n", Role);
      delay(500);
      break;
  }
  LoopMillisNd = millis();
}

// init the stuff
void setup()
{
  // debug line
  Serial.begin(9600);
  Serial.setTimeout(1);
  randomSeed( analogRead(A0));

  EpromSanity();
  EpromDump();

  // RS485
  //RS485_Begin( 28800, 2, 3, 10); // RX 2, TX 3, Enable 10 (so we are compatible with leonardo)
  RS485_Line.begin( 9600); // RX 2, TX 3, Enable 10 (so we are compatible with leonardo)
  RS485_Line.setTimeout(1);
  pinMode(ENABLE_PIN, OUTPUT);  // driver output enable
  digitalWrite ( ENABLE_PIN, LOW);  // disable sending  

  pinMode( 13, OUTPUT);

  InitByRole();
  Activity();
}

// endless call to loop for endless loop
void loop()
{
  unsigned char Ch;

  // about command line    
  Ch = dbgReadCh();
  CmdLineParse( Ch);

  ActByRole();
}

